function [J,E,S,n_iter] = NR_wardhale(S_star,E_star,Y,E_0,idx,Parameters)
%
% INPUT
% - Y           nodal admittance matrix
% - S_star      given complex powers (active/reactive powers)
% - E_star      given voltage magnitudes
% - E_0         initial voltages (phasors)
% - idx.slack   index of the slack bus
% - idx.pq      indices of the PQ buses
% - idx.pv      indices of the PV buses
% - Parameters.tol         tolerance for convergence criterion
% - Parameters.n_max       maximum number of iterations
%
% OUTPUT
% - E           solution voltages (phasors)
% - J           Jacobian at the solution
% - n_iter      number of iterations

n_nodes = length(E_0);
G = real(Y);
B = imag(Y);
    
% Initialization
Ere = real(E_0);
Eim = imag(E_0);
J = [];

for k=1:Parameters.n_max
    n_iter = k;
    
    % Compute nodal voltages/currents/powers
    E = complex(Ere,Eim);
    I = Y*E;
    S = E.*conj(I);
    
    %% Mismatch calculation
    
    % Compute the mismatches for the entire network.
    dS = S_star-S;
    dP = real(dS);
    dQ = imag(dS);
    dV2 = E_star.^2-abs(E).^2; 
    
    % Keep only the relevant mismatches.
    dP(idx.slack) = [];
    dQ(sort([idx.pv;idx.slack])) = [];
    dV2(sort([idx.pq;idx.slack])) = [];
    
    dF = [dP;dQ;dV2]; % mismatch of the power flow equations
    
    %% Convergence check
    
    if(max(abs(dF))<Parameters.tol)
        disp('NR algorithm has converged to a solution!');
        break;
    elseif(k==Parameters.n_max)
        disp('NR algorithm reached the maximum number of iterations!');
    end
   
    
    %%  Jacobian construction
    
    % For the sake of simplicity, the blocks of J are constructed
    % for the whole network (i.e., with size n_nodes x n_nodes).
    % The unnecessary rows/columns are removed subsequently
    
    % Extract real/imaginary part
    Ere = real(E);
    Eim = imag(E);
    
    % Initialization
    J_PR = zeros(n_nodes,n_nodes); % derivative: P versus E_re
    J_PX = zeros(n_nodes,n_nodes); % derivative: P versus E_im
    J_QR = zeros(n_nodes,n_nodes); % derivative: Q versus E_re
    J_QX = zeros(n_nodes,n_nodes); % derivative: Q versus E_im
    J_ER = zeros(n_nodes,n_nodes); % derivative: E^2 versus E_re
    J_EX = zeros(n_nodes,n_nodes); % derivative: E^2 versus E_im
    
    % Construction
    for i=1:n_nodes
        
        % Diagonal elements (terms outside the sum)
        J_PR(i,i) = 2*G(i,i)*Ere(i);
        J_PX(i,i) = 2*G(i,i)*Eim(i);
        J_QR(i,i) = -2*B(i,i)*Ere(i);
        J_QX(i,i) = -2*B(i,i)*Eim(i);
        J_ER(i,i) = 2*Ere(i);
        J_EX(i,i) = 2*Eim(i);
        
        for j=1:n_nodes
            if(j~=i)
                % Diagonal elements (terms inside the sum)
                J_PR(i,i) = J_PR(i,i) + G(i,j)*Ere(j) - B(i,j)*Eim(j);
                J_PX(i,i) = J_PX(i,i) + B(i,j)*Ere(j) + G(i,j)*Eim(j);
                J_QR(i,i) = J_QR(i,i) - B(i,j)*Ere(j) - G(i,j)*Eim(j);
                J_QX(i,i) = J_QX(i,i) + G(i,j)*Ere(j) - B(i,j)*Eim(j);
                
            end
        end
    end
    
    % Remove extra rows (i.e., unnecessary equations)
    % slack bus: P & Q & E^2, PV buses: Q, PQ buses: E^2
    
    J_PR(idx.slack,:) = [];
    J_PX(idx.slack,:) = [];
    
    J_QR([idx.pv;idx.slack],:) = [];
    J_QX([idx.pv;idx.slack],:) = [];
    
    J_ER([idx.pq;idx.slack],:)=[];
    J_EX([idx.pq;idx.slack],:)=[];
    
    % Remove extra columns (i.e., variables)
    % slack bus: E_re & E_im
    
    J_PR(:,idx.slack)=[];
    J_QR(:,idx.slack)=[];
    J_ER(:,idx.slack)=[];
    
    J_PX(:,idx.slack)=[];
    J_QX(:,idx.slack)=[];
    J_EX(:,idx.slack)=[];
    
    % Combination
    J = [J_PR,J_PX;J_QR,J_QX;J_ER,J_EX];
    
    %% Solution update
    
    % Solve
    dx = J \ dF;
    
    % Reconstruct the solution
    
    dEre = zeros(length(Ere),1);
    dEre(sort([idx.pq;idx.pv]),1) = dx(1:(length(idx.pq)+length(idx.pv)));
    
    dEim = zeros(length(Eim),1);
    dEim(sort([idx.pq;idx.pv]),1) = dx((length(idx.pq)+length(idx.pv)+1):end);
    
    % Update
    Ere = Ere + dEre;
    Eim = Eim + dEim;
end

E = complex(Ere,Eim);

end